#ifndef BOHA_UTIL
#define BOHA_UTIL

#include "config.h"

#include "cpputil.h"

#include <SDL.h>

#include <string>
#include <stdexcept>
#include <cmath>
#include <iostream>
#include <cassert>

#include <boost/lexical_cast.hpp>

const double PI = 3.1415926;
const double SQRT_TWO = std::sqrt(2.0);

const double PI_PER_2 = PI / 2;
const double PI_PER_4 = PI / 4;
const double PI_PER_8 = PI / 8;

double dtor(double x);
double rtod(double x);

template <typename T_>
inline T_ absolute(T_ v) {
	return (v < 0) ? -v : v;
}

void ignoreBrankAndComment(std::string& str, std::string delim = "#");

class PolarCoord;

struct GetDoubleCastPolicy {
	template <typename T_>
	static double getDouble(T_ v) { return static_cast<double>(v); }
};

struct GetDoubleMethodPolicy {
	template <typename T_>
	static double getDouble(T_ v) { return v.getDouble(); }
};

template <typename T_, class GetDoublePolicy_ =GetDoubleCastPolicy>
struct Coord
{
	T_ x, y;

	Coord() {}
	Coord(T_ x0, T_ y0) : x(x0), y(y0) {}
	template <typename T2_, class GetDoublePolicy2_>
	Coord(const Coord<T2_, GetDoublePolicy2_>& rhs) : x(rhs.x), y(rhs.y) {}
	explicit Coord(const std::string& rhs);
    explicit Coord(const PolarCoord& rhs)
		: x(std::sin(rhs.r)*rhs.l), y(-std::cos(rhs.r)*rhs.l)
		{}

    const Coord operator+ (const Coord& rhs) const {
		Coord p(x+rhs.x, y+rhs.y);
		return p;
    }
    const Coord operator- (const Coord& rhs) const {
		Coord p(x-rhs.x, y-rhs.y);
		return p;
    }
    const Coord operator* (const T_ rhs) const {
		Coord p(x*rhs, y*rhs);
		return p;
    }
    const Coord operator/ (const T_ rhs) const {
		Coord p(x/rhs, y/rhs);
		return p;
    }
    const Coord& operator+= (const Coord& rhs) {
		x += rhs.x;
		y += rhs.y;
		return *this;
    }
    const Coord& operator-= (const Coord& rhs) {
		x -= rhs.x;
		y -= rhs.y;
		return *this;
    }
    const Coord& operator*= (const T_ rhs) {
		x *= rhs;
		y *= rhs;
		return *this;
    }
    const Coord& operator/= (const T_ rhs) {
		x /= rhs;
		y /= rhs;
		return *this;
    }
    const Coord operator-() {
		Coord p(-x, -y);
		return p;
    }

	friend std::ostream& operator<< (std::ostream& os, const Coord<T_>& rhs) {
		os << '(' << rhs.x << ',' << rhs.y << ')';
		return os;
	}

	T_ length2() const {
		return x*x+y*y;
	}
	T_ length2(const Coord& p) const {
		T_ x1 = x-p.x;
		T_ y1 = y-p.y;
		return x1*x1+y1*y1;
	}

    double length() const {
		return std::sqrt(GetDoublePolicy_::getDouble(length2()));
	}
    double length(const Coord& p) const {
		return std::sqrt(GetDoublePolicy_::getDouble(length2(p)));
	}

	void rotate(double angle);

    void unit();

	/**
	 * オ 0xŉEB
	 */
    double angle() const;
    double angle(const Coord& p) const {
		Coord vec(*this - p);
		return vec.angle();
	}

    double innerProduct() const {
		return x*x+y*y;
    }

    int xi() const { return static_cast<int>(x); }
    int yi() const { return static_cast<int>(y); }
	double xd() const { return GetDoublePolicy_::getDouble(x); }
	double yd() const { return GetDoublePolicy_::getDouble(y); }

};

typedef Coord<Sint16> ICoord;
typedef Coord<double> Point;

class PolarCoord {
public:
    PolarCoord();
    PolarCoord(const PolarCoord& rhs);
    PolarCoord(double l0, double r0);
    explicit PolarCoord(const Point& pnt);

    double l, r;
};

class Rect {
public:
    Rect ();
    Rect (const Rect & rhs);
    // if 'to' is true, w = xe 
    Rect (const int xs, const int ys, const int xe, const int ye,
		  const bool to = false);
    Rect (const Point& ps, const Point& pe, const bool to = false);

    SDL_Rect* val() { return &val_; }
    const SDL_Rect* val() const { return &val_; }

    Sint16 x() const { return val_.x; }
	Sint16 y() const { return val_.y; }
	Sint16 w() const { return val_.w; }
	Sint16 h() const { return val_.h; }
	Sint16 x2() const { return x()+w()-1; }
	Sint16 y2() const { return y()+h()-1; }
	Sint16 cx() const { return val_.x + val_.w / 2; }
	Sint16 cy() const { return val_.y + val_.h / 2; }
	Point start() const { return Point(x(), y()); }
	Point end() const { return Point(x2(), y2()); }
	Point size() const { return Point(w(), h()); }
	Point center() const { return Point(cx(), cy()); }

private:
    SDL_Rect val_;
};

class Color {
public:
    Color(int r, int g, int b) {
		col_.r = r;
		col_.g = g;
		col_.b = b;
	}

	int r() const { return col_.r; }
	int g() const { return col_.g; }
	int b() const { return col_.b; }

	SDL_Color col() const { return col_; }

	Uint32 map(SDL_PixelFormat* fmt) {
		return SDL_MapRGB(fmt, col_.r, col_.g, col_.b);
	}

public:
	static const Color RED, ORANGE, YELLOW, GREEN, BLUE, PURPLE;
	static const Color WHITE, BLACK, GRAY;

protected:
	SDL_Color col_;
};

inline std::ostream& operator<< (std::ostream& os, const Color& c) {
	os << '(' << c.r() << ',' << c.g() << ',' << c.b() << ')';
	return os;
}

template <typename T_, class GetDoublePolicy_>
Coord<T_, GetDoublePolicy_>::Coord(const std::string& rhs) {
	if (rhs.empty()) {
		x = 0;
		y = 0;
		return;
	}

	size_t index = rhs.find('x');
	if (index == std::string::npos) index = rhs.find(' ');
	assert(index != std::string::npos);

	x = boost::lexical_cast<int>(rhs.substr(0, index));
	y = boost::lexical_cast<int>(rhs.substr(index+1));
}

template <typename T_, class GetDoublePolicy_>
double Coord<T_, GetDoublePolicy_>::angle() const {
	double ret;
    if (y == 0) {
		if (x > 0) return PI_PER_2;
		else return PI_PER_2 * 3;
	}
    else ret = std::atan(GetDoublePolicy_::getDouble(x/y));
	return (y > 0) ? PI-ret : -ret;
}

template <typename T_, class GetDoublePolicy_>
void Coord<T_, GetDoublePolicy_>::unit() {
	T_ x2 = absolute(x) * x;
	T_ y2 = absolute(y) * y;
    T_ x2y2 = absolute(x2)+absolute(y2);
    x = x2 / x2y2;
    y = y2 / x2y2;
}

template <typename T_, class GetDoublePolicy_>
void Coord<T_, GetDoublePolicy_>::rotate(double angle) {
	double cs = cos(angle);
	double sn = sin(angle);
	T_ x2 = cs * x - sn * y;
	y = sn * x + cs * y;
	x = x2;
}

#endif // BOHA_UTIL
