#ifndef __KEYBOARDWM_LOADCONF_H__
#define __KEYBOARDWM_LOADCONF_H__

#include <fstream>
#include <string>
#include <stdexcept>

#include <sstream>

#include <boost/lexical_cast.hpp>

class ConfMapGenerater {
public:
    ConfMapGenerater();

    /// load "key = val" or "key val" style config file.
    /**
     * @param map
     *  STL std::map<Key, Val>
     *  class Key and Val must be able to change from std::string,
     *  generally, Key and Val are std::string.
     */
    //@{
    /**
     * @param str config file's path
     */
    template <class Map_>
    void generate(const std::string& str, Map_& map);
    /**
     * @param is config stream
     */
    template <class Map_>
    void generate(std::istream& is, Map_& map);
    //@}

	template <class Map_>
	void save(const std::string& str, const Map_& map);
	template <class Map_>
	void save(std::ostream& is, const Map_& map);

public:
    void setReportDevice(std::ostream& os) { report_ = &os; }

private:
    std::ostream* report_;
};

template<class Map_>
void ConfMapGenerater::generate(const std::string& str, Map_& map) {
    std::ifstream is(str.c_str());
    if (!is) {
		if (report_) {
			*report_ << str
					 << ": file not found in generateConfMap" << std::endl;
		}
    }
    else {
		generate(is, map);
    }
}

template <class Map_>
void ConfMapGenerater::save(const std::string& str, const Map_& map) {
	std::ofstream os(str.c_str());
	if (!os) {
		if (report_) {
			*report_ << str
					 << ": file cannot opne in saveConfMap" << std::endl;
		}
	}
	else {
		save(os, map);
	} 
}

template<class Map_>
void ConfMapGenerater::generate(std::istream& is, Map_& map) {
    static const int MAXBUF = 256;
    char ln[MAXBUF];

    while (is.getline(ln, MAXBUF)) {
		std::string line(ln);
		unsigned int s;

		// Rg̏
		if ((s = line.find("#")) != std::string::npos)
			line.erase(s);

		// s󔒂̏
		if (!line.empty() && line[0] == ' ')
			line.erase(0, line.find_first_not_of(' '));

		std::istringstream ss(line);
		std::string key, val;
		ss >> key >> val;
		if (val == "=") ss >> val;

		while (ss) {
			std::string add;
			ss >> add;
			if (add != "") val += " " + add;
		}

		if (key != "") {
			try {
				map[key] = val;
			}
			catch (std::runtime_error re) {
				if (report_ != 0) *report_ << re.what() << std::endl;
			}
		}
    }
}

template <class Map_>
void ConfMapGenerater::save(std::ostream& os, const Map_& map) {
	for (typename Map_::const_iterator ite = map.begin();
		 ite != map.end(); ++ite)
	{
		os << ite->first << " = " << ite->second << std::endl;
	}
}

/// generate ̃wpQBK؂Ȍ^Ŏ肾B

/// bool, ftHg false
template <class Map_>
inline bool getBoolFromMap(const Map_& map, const std::string& str) {
	typename Map_::const_iterator ite = map.find(str);
	if (ite == map.end() || ite->second == "" ||
		ite->second == "false" || ite->second == "off" ||
		ite->second == "0")
	{
		return false;
	}
	return true;
}

/// int, ftHg 0
template <class Map_>
inline int getIntFromMap(const Map_& map, const std::string& str) {
	typename Map_::const_iterator ite = map.find(str);
	if (ite == map.end())
	{
		return 0;
	}
	return boost::lexical_cast<int>(ite->second);
}

/// size_t, ftHg 0
template <class Map_>
inline int getSizeFromMap(const Map_& map, const std::string& str) {
	typename Map_::const_iterator ite = map.find(str);
	if (ite == map.end())
	{
		return 0;
	}
	return boost::lexical_cast<size_t>(ite->second);
}

/// double, ftHg 0.0
template <class Map_>
inline double getDoubleFromMap(const Map_& map, const std::string& str) {
	typename Map_::const_iterator ite = map.find(str);
	if (ite == map.end())
	{
		return 0;
	}
	return boost::lexical_cast<double>(ite->second);
}


/// string, ftHg ""
template <class Map_>
inline std::string getStringFromMap(const Map_& map, const std::string& str) {
	typename Map_::const_iterator ite = map.find(str);
	if (ite == map.end())
	{
		return "";
	}
	return ite->second;
}

template <class Map_>
inline bool isInMap(const Map_& map, const std::string& str) {
	typename Map_::const_iterator ite = map.find(str);
	if (ite == map.end())
	{
		return false;
	}
	return true;
}

#endif // ! __KEYBOARDWM_LOADCONF_H__

