#include "surface.h"
#include "game.h"

#include <cctype>
#include <cmath>
#include <vector>

Uint32 Surface::getPixel(int x, int y) {
    int bpp = surf_->format->BytesPerPixel;
    /* Here p is the address to the pixel we want to retrieve */
    Uint8 *p = (Uint8 *)surf_->pixels + y * surf_->pitch + x * bpp;

    switch(bpp) {
    case 1:
        return *p;

    case 2:
        return *(Uint16 *)p;

    case 3:
        if(SDL_BYTEORDER == SDL_BIG_ENDIAN)
            return p[0] << 16 | p[1] << 8 | p[2];
        else
            return p[0] | p[1] << 8 | p[2] << 16;

    case 4:
        return *(Uint32 *)p;

    default:
        return 0;       /* shouldn't happen, but avoids warnings */
    }

}

void Surface::putPixel(int x, int y, Uint32 pixel) {
    int bpp = surf_->format->BytesPerPixel;
    /* Here p is the address to the pixel we want to set */
    Uint8 *p = (Uint8 *)surf_->pixels + y * surf_->pitch + x * bpp;

    switch(bpp) {
    case 1:
        *p = pixel;
        break;

    case 2:
        *(Uint16 *)p = pixel;
        break;

    case 3:
        if(SDL_BYTEORDER == SDL_BIG_ENDIAN) {
            p[0] = (pixel >> 16) & 0xff;
            p[1] = (pixel >> 8) & 0xff;
            p[2] = pixel & 0xff;
        } else {
            p[0] = pixel & 0xff;
            p[1] = (pixel >> 8) & 0xff;
            p[2] = (pixel >> 16) & 0xff;
        }
        break;

    case 4:
        *(Uint32 *)p = pixel;
        break;
    }

}

Surface::Surface(const Surface& rhs)
	: size_(rhs.size()) {
	surf_ = SDL_ConvertSurface(rhs.surf_, rhs.surf_->format, SDL_SWSURFACE);
	assert(surf_ != 0);

	calcCache();
}

Surface::Surface(int x, int y, int bpp, Uint32 flag,
				 Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask)
	: size_(x, y) {
    surf_ = SDL_CreateRGBSurface(flag, x, y, bpp, Rmask, Gmask, Bmask, Amask);
    assert(surf_ != 0);

	calcCache();
}

Surface::Surface(const Point& size, int bpp, Uint32 flag,
				 Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask)
	: size_(size) {
	surf_ = SDL_CreateRGBSurface(flag, size.xi(), size.yi(), bpp,
								 Rmask, Gmask, Bmask, Amask);
	assert(surf_ != 0);

	calcCache();
}

Surface::Surface(Surface& src, const Rect& rect)
	: size_(rect.size())
{
	SDL_Surface* s = src.surf_;
    int bpp = s->format->BytesPerPixel;
    Uint8* pixel = (Uint8 *)s->pixels + rect.y() * s->pitch + rect.x() * bpp;
	surf_ = SDL_CreateRGBSurfaceFrom(pixel, rect.w(), rect.h(),
									 bpp*8, s->pitch , 0 ,0, 0, 0);
}

void Surface::printInformation(std::ostream& os) const {
    os << surf_->w << "*" << surf_->h << "*"
       << (int)surf_->format->BitsPerPixel << "\n";
}

void Surface::putString(const std::string& str, const Point& p,
						const Color& col) {
	Kanji_Font* font = Game::instance()->getFont();
	Kanji_PutText(font, p.xi(), p.yi(), surf_, str.c_str(), col.col());
}

int Surface::fontHeight() {
	return Kanji_FontHeight(Game::instance()->getFont());
}
int Surface::fontWidth() {
	return Kanji_FontWidth(Game::instance()->getFont(), 0); }
int Surface::fontWidth(const std::string& str) {
	return Kanji_FontWidth(Game::instance()->getFont(), str.c_str());
}

void Surface::convertDisplayFormat() {
	SDL_Surface* converted = SDL_DisplayFormat(surf_);
	assert(converted != 0);
	surf_ = converted;
}

void Surface::setMask(Color col) {
	double rmask = static_cast<double>(col.r()) / 0xff;
	double gmask = static_cast<double>(col.g()) / 0xff;
	double bmask = static_cast<double>(col.b()) / 0xff;

	lock();
	for (int x = 0; x < surf_->w; x++) {
		for (int y = 0; y < surf_->h; y++) {
			Uint32 pixel = getPixel(x, y);
			if (pixel != surf_->format->colorkey) {
				Color before = getRGB(pixel);
				Color after(static_cast<int>(before.r()*rmask),
							static_cast<int>(before.g()*gmask),
							static_cast<int>(before.b()*bmask));
				putPixel(x, y, mapRGB(after));
			}
		}
	}
	unlock();
}

Surface::Surface(const std::string& str) {
    surf_ = SDL_LoadBMP(str.c_str());
    check(surf_ != 0, str + " not found");

    size_ = Point(surf_->w, surf_->h);

	calcCache();
}

Surface::~Surface() {
    if (surf_ != 0) {
		SDL_FreeSurface(surf_);
	}
}

void Surface::calcCache() {
	halfDiagonal_ = size_.length() / 2.0;
	halfSize_ = size_ / 2;
}

void Surface::drawHLine(Sint16 x1, Sint16 x2, Sint16 y, Color col) {
	if(x1>x2){Sint16 tmp=x1; x1=x2; x2=tmp;}

	fillRect(Rect(x1, y, x2, y), col);
}

void Surface::drawVLine(Sint16 x, Sint16 y1, Sint16 y2, Color col) {
	if(y1>y2){Sint16 tmp=y1; y1=y2; y2=tmp;}

	fillRect(Rect(x, y1, x, y2), col);
}

void Surface::drawRect(Sint16 x1, Sint16 y1, Sint16 x2, Sint16 y2, Color col) {
	drawHLine(x1, x2, y1, col);
	drawHLine(x1, x2, y2, col);
	drawVLine(x1, y1, y2, col);
	drawVLine(x2, y1, y2, col);
}

#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
        #define sge_clip_xmin(pnt) pnt->clip_rect.x
        #define sge_clip_xmax(pnt) pnt->clip_rect.x + pnt->clip_rect.w-1
        #define sge_clip_ymin(pnt) pnt->clip_rect.y
        #define sge_clip_ymax(pnt) pnt->clip_rect.y + pnt->clip_rect.h-1
#else
        #define sge_clip_xmin(pnt) pnt->clip_minx
        #define sge_clip_xmax(pnt) pnt->clip_maxx
        #define sge_clip_ymin(pnt) pnt->clip_miny
        #define sge_clip_ymax(pnt) pnt->clip_maxy
#endif

void Surface::drawLine(Sint16 x1, Sint16 y1, Sint16 x2, Sint16 y2, Color col) {
	Uint32 color = mapRGB(col);

	/* Clipping */
	if(x2<sge_clip_xmin(surf_) || x1>sge_clip_xmax(surf_) ||
	   y2<sge_clip_ymin(surf_) || y1>sge_clip_ymax(surf_))
		return;
	if (x1 < sge_clip_xmin(surf_))
		x1 = sge_clip_xmin(surf_);
	if (x2 > sge_clip_xmax(surf_))
		x2 = sge_clip_xmax(surf_);
	if (y1 < sge_clip_ymin(surf_))
		y1 = sge_clip_ymin(surf_);
	if (y2 > sge_clip_ymax(surf_))
		y2 = sge_clip_ymax(surf_);

	Sint16 dx, dy, sdx, sdy, x, y;

	dx = x2 - x1;
	dy = y2 - y1;

	sdx = (dx < 0) ? -1 : 1;
	sdy = (dy < 0) ? -1 : 1;

	dx = sdx * dx + 1;
	dy = sdy * dy + 1;
	x = y = 0;
        
	Sint16 pixx = surf_->format->BytesPerPixel;
	Sint16 pixy = surf_->pitch;
	Uint8 *pixel = (Uint8*)surf_->pixels + y1*pixy + x1*pixx;

	pixx *= sdx;
	pixy *= sdy;

	if (dx < dy) {
		Sint32 tmp = dx; dx = dy; dy = Sint16(tmp);
		tmp = pixx; pixx = pixy; pixy = tmp;
	}       
        
	switch(surf_->format->BytesPerPixel) {
	case 2: {
		for(x=0; x < dx; x++) {
			*(Uint16*)pixel = color;
                                
			y += dy; 
			if (y >= dx) {
				y -= dx; 
				pixel += pixy;
			}
			pixel += pixx;
		}
	}
		break;
	case 4: {
		for(x=0; x < dx; x++) {
			*(Uint32*)pixel = color;
                                
			y += dy; 
			if (y >= dx) {
				y -= dx; 
				pixel += pixy;
			}
			pixel += pixx;
		}
	}
		break;

	default:
		assert(0);
	}

	
}

