#include "mysdl_padinput.h"
#include "loadconf.h"

#include <boost/lexical_cast.hpp>

#include <algorithm>
#include <functional>
#include <sstream>
#include <string>
#include <vector>
#include <iterator>

using namespace MSDL;

const int Axis::NONE = 0;
const int Axis::UP = 1;
const int Axis::DOWN = 2;
const int Axis::RIGHT = 4;
const int Axis::LEFT = 8;
const int Axis::UPRIGHT = UP | RIGHT;
const int Axis::UPLEFT = UP | LEFT;
const int Axis::DOWNRIGHT = DOWN | RIGHT;
const int Axis::DOWNLEFT = DOWN | LEFT;

const int JoystickAxisButtonInput::JOYSTICK_PLAY_WIDTH = 8192;

int Axis::getSmallAxisCode() const {
	if (axis_ == NONE) return 0;
	else if (axis_ == UP) return 1;
	else if (axis_ == UPRIGHT) return 2;
	else if (axis_ == RIGHT) return 3;
	else if (axis_ == DOWNRIGHT) return 4;
	else if (axis_ == DOWN) return 5;
	else if (axis_ == DOWNLEFT) return 6;
	else if (axis_ == LEFT) return 7;
	else if (axis_ == UPLEFT) return 8;
	else return 0;
/* if pad is keyboard, state right and left is allowed.
	else throw Error(
		"getSmallAxisCode: broken axis: " +
		boost::lexical_cast<std::string>(axis_));
*/
}

Axis Axis::createFromSmallCode(int code) {
	if (code == 0) return Axis(NONE);
	else if (code == 1) return Axis(UP);
	else if (code == 2) return Axis(UPRIGHT);
	else if (code == 3) return Axis(RIGHT);
	else if (code == 4) return Axis(DOWNRIGHT);
	else if (code == 5) return Axis(DOWN);
	else if (code == 6) return Axis(DOWNLEFT);
	else if (code == 7) return Axis(LEFT);
	else if (code == 8) return Axis(UPLEFT);
	else throw Error("createFromSmallCode: unknown code.");
}

bool JoystickAxisButtonInput::getButton() {
	if (axis_.isRight()) {
		if (joystick_->axisX() > JOYSTICK_PLAY_WIDTH) return true;
	}
	else if (axis_.isLeft()) {
		if (joystick_->axisX() < -JOYSTICK_PLAY_WIDTH) return true;
	}
	else if (axis_.isUp()) {
		if (joystick_->axisY() < -JOYSTICK_PLAY_WIDTH) return true;
	}
	else if (axis_.isDown()) {
		if (joystick_->axisY() > JOYSTICK_PLAY_WIDTH) return true;
	}

	return false;
}

Axis JoystickAxisButtonInput::generateAxis(Uint8 axis, Sint16 value) {
	if (axis == 0) {
		if (value > JOYSTICK_PLAY_WIDTH) return Axis(Axis::RIGHT);
		else if (value < JOYSTICK_PLAY_WIDTH) return Axis(Axis::LEFT);
	}
	else if (axis == 1) {
		if (value > JOYSTICK_PLAY_WIDTH) return Axis(Axis::DOWN);
		else if (value < JOYSTICK_PLAY_WIDTH) return Axis(Axis::UP);
	}
	else {
		throwError("JoystickAxisButtonInput::generateFromSDLAxis: ???");
	}
	return Axis();
}

InputState::InputState(InputBase& input)
	: axis_(input.getAxis()), buttonMap_(input.getAllButton().release())
{}

bool InputState::getButton(int id) {
	return buttonMap_->find(id)->second;
}

InputBase::AllButtonPtr InputState::getAllButton() {
	return AllButtonPtr(new AllButton(*buttonMap_.get()));
}

InputMgr::~InputMgr() {
	delete_clear(inputs_);
}

void InputMgr::setDevice(std::auto_ptr<InputBase> input) {
	inputs_.push_back(input.release());
}

Axis InputMgr::getAxis() {
	Axis ret;
	for (std::vector<InputBase*>::const_iterator ite = inputs_.begin();
		 ite != inputs_.end(); ite++)
	{
		ret.add((*ite)->getAxis());
	}
	return ret;
}

bool InputMgr::getButton(int id) {
	for (std::vector<InputBase*>::const_iterator ite = inputs_.begin();
		 ite != inputs_.end(); ite++)
	{
		if ((*ite)->getButton(id)) return true;
	}
	return false;
}

InputBase::AllButtonPtr InputBase::getAllButton() {
	AllButtonPtr ret(new AllButton);
	for (std::vector<int>::const_iterator ite = buttons_.begin();
		 ite != buttons_.end(); ++ite)
	{
		(*ret)[*ite] = getButton(*ite);
	}
	return ret;
}

InputBase::AllButtonPtr InputMgr::getAllButton() {
	AllButtonPtr ret(new AllButton);
	for (std::vector<InputBase*>::const_iterator ite = inputs_.begin();
		 ite != inputs_.end(); ite++)
	{
		const AllButtonPtr buttons((*ite)->getAllButton().release());

		for (AllButton::const_iterator ite2 = buttons->begin();
			 ite2 != buttons->end(); ite2++)
		{

			if (ret->find(ite2->first) == ret->end()) {
				ret->insert(*ite2);
			}
			else {
				(*ret)[ite2->first] |= ite2->second;
			}
		}
	}
	return ret;
}

std::auto_ptr<ButtonInput> MSDL::generateAnyButtonInput() {
	// ̊֐̎͂ւ傷

	typedef std::auto_ptr<ButtonInput> BIP;

	SDL_Event ev;
	while (1) {
		while (SDL_PollEvent(&ev)) {
			if (ev.type == SDL_QUIT) {
				return BIP(0);
			}
			else if (ev.type == SDL_KEYDOWN) {
				return BIP(new KeyButtonInput(ev.key.keysym.sym));
			}
			else if (ev.type == SDL_JOYAXISMOTION) {
				SDL_JoyAxisEvent* je = &ev.jaxis;
				if (JoystickMgr::instance()->getJoystick(je->which) != 0) {
					Axis axis = 
						JoystickAxisButtonInput::generateAxis(
							je->axis, je->value);
					return BIP(new JoystickAxisButtonInput(je->which, axis));
				}
			}
			else if (ev.type == SDL_JOYBUTTONDOWN) {
				SDL_JoyButtonEvent* je = &ev.jbutton;
				if (JoystickMgr::instance()->getJoystick(je->which) != 0) {
					return BIP(new JoystickButtonInput(je->which, je->button));
				}
			}
		}
	}
}

std::auto_ptr<ButtonInput>
MSDL::generateButtonInputFromString(const std::string& str) {
	typedef std::auto_ptr<ButtonInput> BIP;

	unsigned int unsco = str.find('_') + 1;
	std::string prefix(str.substr(0, unsco));
	std::string val(str.substr(unsco));

	if (prefix == "KEY_") {
		SDLKey key;

		if (val.size() != 1) {
			if (val == "tab") key = SDLK_TAB;
			else if (val == "space") key = SDLK_SPACE;
			else if (val == "delete") key = SDLK_DELETE;
			else if (val == "return") key = SDLK_RETURN;
			else if (val == "escape") key = SDLK_ESCAPE;
			else if (val == "up") key = SDLK_UP;
			else if (val == "down") key = SDLK_DOWN;
			else if (val == "left") key = SDLK_LEFT;
			else if (val == "right") key = SDLK_RIGHT;
			else if (val == "insert") key = SDLK_INSERT;
			else if (val == "rshift") key = SDLK_RSHIFT;
			else if (val == "ralt") key = SDLK_RALT;
			else if (val == "rctrl") key = SDLK_RCTRL;
			else if (val == "lshift") key = SDLK_LSHIFT;
			else if (val == "lalt") key = SDLK_LALT;
			else if (val == "lctrl") key = SDLK_LCTRL;
			else if (val == "f11") key = SDLK_F11;
			else if (val == "pageup") key = SDLK_PAGEUP;
			else if (val == "pagedown") key = SDLK_PAGEDOWN;
			else throwError(val + ": unknown key");
		}
		else {
			key = static_cast<SDLKey>(val[0]);
		}

		return BIP(new KeyButtonInput(key));
	}
	else if (prefix == "JOY_" || prefix == "JOY1_" || prefix == "JOY2_") {
		int joynum = (prefix == "JOY2_") ? 1 : 0;
		if (JoystickMgr::instance()->getJoystick(joynum)) {
			if (val.size() != 1) {
				Axis axis;
				if (val == "UP") axis = Axis::UP;
				else if (val == "RIGHT") axis = Axis::RIGHT;
				else if (val == "DOWN") axis = Axis::DOWN;
				else if (val == "LEFT") axis = Axis::LEFT;
				else throwError(val + ": unknown key");

				return BIP(new JoystickAxisButtonInput(joynum, axis));
			}
			else {
				std::istringstream iss(val);
				int index;
				iss >> index;

				return BIP(new JoystickButtonInput(joynum, index));
			}
		}
		else {
			return BIP(0);
		}
	}
	else {
		throwError("generateButtonInputFromString: unknown string format");
	}

	return BIP(0);
}

std::auto_ptr<AxisInput> MSDL::generateKeyAxis() {
	return std::auto_ptr<AxisInput>(
		new ButtonAxisInput(
			std::auto_ptr<ButtonInput>(new KeyButtonInput(SDLK_UP)),
			std::auto_ptr<ButtonInput>(new KeyButtonInput(SDLK_RIGHT)),
			std::auto_ptr<ButtonInput>(new KeyButtonInput(SDLK_DOWN)),
			std::auto_ptr<ButtonInput>(new KeyButtonInput(SDLK_LEFT))));
}

std::auto_ptr<ButtonInput> generateJoyAxisHelper(int i, Axis axis) {
	return std::auto_ptr<ButtonInput>(new JoystickAxisButtonInput(i, axis));
}

std::auto_ptr<AxisInput> MSDL::generateJoyAxis(int index) {
	return std::auto_ptr<AxisInput>(
		new ButtonAxisInput(
			generateJoyAxisHelper(index, Axis(Axis::UP)),
			generateJoyAxisHelper(index, Axis(Axis::RIGHT)),
			generateJoyAxisHelper(index, Axis(Axis::DOWN)),
			generateJoyAxisHelper(index, Axis(Axis::LEFT))));
}

bool InputBase::getAny() {
	if (getAxis() != Axis::NONE) return true;
	return getAnyButton();
}

bool InputBase::getAnyButton() {
	AllButtonPtr abp(getAllButton().release());
	for (AllButton::iterator ite = abp->begin(); ite != abp->end(); ++ite) {
		if (ite->second) return true;
	}

	return false;
}

Axis Input::getAxis() {
	Axis ret;
	for (std::vector<AxisInput*>::const_iterator ite = axis_.begin();
		 ite != axis_.end(); ite++)
	{
		ret.add((*ite)->getAxis());
	}
	return ret;
}

bool Input::getButton(int id) {
	ButtonMap::const_iterator begin = buttonMap_.lower_bound(id);
	ButtonMap::const_iterator end = buttonMap_.upper_bound(id);
	for (ButtonMap::const_iterator ite = begin; ite != end; ++ite) {
		if (ite->second->getButton()) return true;
	}

	return false;
}

InputBase::AllButtonPtr Input::getAllButton() {
	AllButtonPtr ret(new AllButton);
	for (ButtonMap::const_iterator ite = buttonMap_.begin();
		 ite != buttonMap_.end(); ite++)
	{
		if (ret->find(ite->first) == ret->end()) {
			ret->insert(std::make_pair(ite->first, getButton(ite->first)));
		}
	}
	return ret;
}

Axis ButtonAxisInput::getAxis() {
	Axis ret;

	for (std::vector<ButtonInput*>::const_iterator ite = up_.begin();
		 ite != up_.end(); ite++)
	{
		if ((*ite)->getButton()) {
			ret.add(Axis::UP);
			break;
		}
	}

	for (std::vector<ButtonInput*>::const_iterator ite = right_.begin();
		 ite != right_.end(); ite++)
	{
		if ((*ite)->getButton()) {
			ret.add(Axis::RIGHT);
			break;
		}
	}

	for (std::vector<ButtonInput*>::const_iterator ite = down_.begin();
		 ite != down_.end(); ite++)
	{
		if ((*ite)->getButton()) {
			ret.add(Axis::DOWN);
			break;
		}
	}

	for (std::vector<ButtonInput*>::const_iterator ite = left_.begin();
		 ite != left_.end(); ite++)
	{
		if ((*ite)->getButton()) {
			ret.add(Axis::LEFT);
			break;
		}
	}

	return ret;
}

std::string KeyButtonInput::getName() const {
	std::string sdlk = SDL_GetKeyName(key_);
	if (sdlk.length() > 6 && sdlk.substr(0, 6) == "right ") {
		sdlk.erase(1, 6);
	}
	if (sdlk.length() > 5 && sdlk.substr(0, 5) == "left ") {
		sdlk.erase(1, 5);
	}
	return std::string("KEY_") + sdlk;
}

std::string JoystickButtonInput::getName() const {
	std::ostringstream oss;
	oss << "JOY_" << button_;
	return oss.str();
}

std::string JoystickAxisButtonInput::getName() const {
	return std::string("JOY_") + axis_.getName();
}

std::string Axis::getName() const {
	if (axis_ == NONE) return "NONE";
	else if (axis_ == UP) return "UP";
	else if (axis_ == UPRIGHT) return "UPRIGHT";
	else if (axis_ == RIGHT) return "RIGHT";
	else if (axis_ == DOWNRIGHT) return "DOWNRIGHT";
	else if (axis_ == DOWN) return "DOWN";
	else if (axis_ == DOWNLEFT) return "DOWNLEFT";
	else if (axis_ == LEFT) return "LEFT";
	else if (axis_ == UPLEFT) return "UPLEFT";
	else return "UNKNOWN";
}

namespace {
	void separateStrs(const std::string& str,
					  std::vector<std::string>& strs) 
	{
		std::istringstream iss(str);
		std::copy(std::istream_iterator<std::string>(iss),
				  std::istream_iterator<std::string>(),
				  std::back_inserter(strs));
	}
}

std::auto_ptr<Input> MSDL::generateInputFromConf(const std::string& filename) {
	std::auto_ptr<Input> ret(new Input);

	ConfMapGenerater cmg;
    cmg.setReportDevice(std::cerr);
	std::map<std::string, std::string> tmpMap;
    cmg.generate(filename, tmpMap);

	typedef std::vector<std::string> Strs;

	typedef std::map<std::string, std::vector<std::string> > KeyConfMap;
	KeyConfMap keyConfMap;
	keyConfMap.clear();
	for (std::map<std::string, std::string>::iterator ite = tmpMap.begin();
		 ite != tmpMap.end(); ++ite)
	{
		KeyConfMap::value_type elem(ite->first, Strs());
		separateStrs(ite->second, elem.second);
		keyConfMap.insert(elem);
	}

	MSDL::ButtonAxisInput* axisInput = new MSDL::ButtonAxisInput;
	{
		Strs& vals = keyConfMap["UP_AXIS"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				axisInput->setUpAxis(bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["RIGHT_AXIS"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				axisInput->setRightAxis(bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["DOWN_AXIS"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				axisInput->setDownAxis(bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["LEFT_AXIS"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				axisInput->setLeftAxis(bi);
			}
		}
	}
	ret->setAxis(std::auto_ptr<MSDL::AxisInput>(axisInput));

	{
		Strs& vals = keyConfMap["BUTTON_1"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				ret->setButton(0, bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["BUTTON_2"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				ret->setButton(1, bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["BUTTON_3"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				ret->setButton(2, bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["BUTTON_4"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				ret->setButton(3, bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["BUTTON_5"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				ret->setButton(4, bi);
			}
		}
	}
	{
		Strs& vals = keyConfMap["END"];
		for (Strs::iterator ite = vals.begin(); ite != vals.end(); ++ite) {
			std::auto_ptr<MSDL::ButtonInput>
				bi(MSDL::generateButtonInputFromString(*ite).release());
			if (bi.get() != 0) {
				ret->setButton(3, bi);
			}
		}
	}

	return ret;
}

