#include "game.h"
#include "util.h"
#include "conf.h"
#include "cpu.h"
#include "char.h"
#include "enemy.h"
#include "player.h"
#include "command.h"
#include "shot.h"
#include "select.h"
#include "bulletinfo.h"
#include "replay.h"
#include "profile.h"
#include "cpuinfo_sdmkun.h"

#include <bulletml/bulletmlparser.h>
#include <bulletml/bulletmlrunner.h>

#include <algorithm>
#include <functional>
#include <iostream>
#include <fstream>
#include <string>
#include <ctime>
#include <sstream>
#include <cassert>

#include <SDL.h>

const double Game::framePerSecond_ = 60.0 / 960.0;

int Game::getFrameTurn() const {
	if (Conf::instance()->isSlow()) {
		return static_cast<int>(
			timer_.getEllapsedTicks() * framePerSecond_
			* Conf::instance()->slow());
	}
	else {
		return static_cast<int>(timer_.getEllapsedTicks() * framePerSecond_);
	}
}

int Game::getTurn() const {
	return timer_.getTurn();
}

int Game::waitAnyInput() {
	int ret = 0;
	while (ret == 0) {
		SDL_Delay(50);
		ret = getAnyInput();
		if (ret == -1) return -1;
	}
	return 0;
}

int Game::getAnyInput() {
	SDL_PumpEvents();
	endEv();
	if (quit_) return -1;
	if (inputMgr_->getButton(2)) return -1;
	if (inputMgr_->getButton(0) || inputMgr_->getButton(1)) return 1;
	return 0;
}

void Game::waitInputRelease() {
	while (inputMgr_->getAnyButton()) {
		SDL_Delay(50);
		SDL_PumpEvents();
	}
}

void Game::endEv() {
	SDL_Event ev;
	while (SDL_PollEvent(&ev)) {
		if (ev.type == SDL_QUIT) {
			end_ = true;
			quit_ = true;
		}
	}
	if (inputMgr_->getButton(2)) end();

	if (Conf::instance()->timeLimit() != -1 &&
		Conf::instance()->timeLimit() < getFrameTurn())
	{
		end_ = true;
		quit_ = true;
	}
}

int Game::title() {
	int ret = 0;
	if (!Conf::instance()->isFastMode()) {
		do {
			select_.reset(new Select(*inputMgr_));
			ret = select_->run();
			if (-1 == ret) return -1;
		} while (ret == 1);
	}
	else {
		select_.reset(new Select(*inputMgr_));
		select_->fastRun();
	}

	reinit();

	return 0;
}

void Game::initBMP() {
	std::string bmp = Conf::instance()->bmpdir();

	playerg_.reset(new Surface(bmp+"/ball_r.bmp"));
	enemygBoss_.reset(new Surface(bmp+"/ball_g.bmp"));
	enemygZako_.reset(new Surface(bmp+"/ball_wg.bmp"));
	enemygBullet_.reset(new Surface(bmp+"/ball_wr.bmp"));
	shotg_.reset(new Surface(bmp+"/ball_w.bmp"));
	shotg2_.reset(new Surface(bmp+"/ball_wb.bmp"));

	Color trans(255, 0, 255);
	playerg_->setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL, trans);
	enemygBoss_->setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL, trans);
	enemygZako_->setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL, trans);
	enemygBullet_->setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL, trans);
	shotg_->setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL, trans);
	shotg2_->setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL, trans);

	if (select_.get() != 0) {
		select_->initBMP();
	}
}

void Game::init () {
	int ret = SDL_Init(SDL_INIT_VIDEO|SDL_INIT_NOPARACHUTE);
	check(ret == 0, "cannot init video\n");
	ret = SDL_InitSubSystem(SDL_INIT_TIMER);
	check(ret == 0, "cannot init timer\n");
	ret = SDL_InitSubSystem(SDL_INIT_JOYSTICK);
	check(ret == 0, "cannot init joystick\n");
	atexit(SDL_Quit);
	SDL_WM_SetCaption("siroi danmakukun", "sdmkun");

	srand(time(0));

	joystickMgr_.reset(new MSDL::JoystickMgr);

	inputMgr_.reset(
		MSDL::generateInputFromConf("save/key.conf").release());

	screen_ = new Screen;
	Select select;
	select.loadConf();
	select.setConf();
//	switchScreenMode(!Conf::instance()->isFastMode());
	screen_->printInformation(std::cout);

//	font_ = Kanji_OpenFont("fonts/K12-1.bdf", 12);
//	Kanji_AddFont(font_, "fonts/6x12.bdf");
	font_ = Kanji_OpenFont("fonts/K12-1_6x12.f1b", 12);
	Kanji_SetCodingSystem(font_, KANJI_SJIS);

	initBMP();

	bulletInfo_.reset(new BulletInfo());

	quit_ = end_ = false;

	std::cout << "init done." << std::endl;
}

void Game::enemyInit() {
	enemyMgr_.init();

	const Conf* conf = Conf::instance();
	int enemys = conf->enemys();

	for (int i = 0; i < enemys; i++) {
		Enemy* enemy;
		enemy = new BossEnemy(Point(150, 100+i*10), enemygBoss_.get(),
							  enemyMgr_.getBehavior());

		enemy_.push_back(enemy);
		chars_.push_back(enemy);
	}
}

void Game::reinit() {
	int fps = Conf::instance()->fps();
	timer_.setFps(fps);
	timer_.start();
	spf_ = 1.0 / fps;
	if (Conf::instance()->isSlow()) {
		spf_ *= Conf::instance()->slow();
	}
	end_ = false;
	quit_ = false;

	bulletInfoTicks_ = 0;
	ellapsedSecOrRank_ = 0;
	enemyNum_ = 1;

	if (Conf::instance()->mode() != 2) {
		nowBullet_ = 0;
	}

	rank_ = Conf::instance()->level()*0.01;

	enemy_.clear();
	shot_.clear();
	delete_clear(chars_);

	if (Conf::instance()->isCpu()) {
		CpuInputBase::setCpuInfomation(
			std::auto_ptr<CpuInfo>(new CpuInfoSdmkun()));
		cpuInput_.reset(CpuInputBase::getDefaultCpu().release());

		player_ = new Player(Point(150, 350), playerg_.get(), *cpuInput_);
	}
	else {
		player_ = new Player(Point(150, 350), playerg_.get(), *inputMgr_);
	}
	chars_.push_back(player_);

	enemyInit();

	if (!Conf::instance()->isReplay()) replay_.reset(new Replay());

	mainFrame()->fill(Color::BLACK);
	screen_->flip();
}

void Game::addShot(const Point& pnt, const Point& spd, const Enemy* enemy) {
	Shot* shot;
	if (enemy->blue()) {
		shot = new Shot(pnt, spd, shotg2_.get());
	}
	else {
		shot = new Shot(pnt, spd, shotg_.get());
	}

	shot_.push_back(shot);
	chars_.push_back(shot);

	if (Conf::instance()->isCpu()) {
		cpuInput_->registShot(std::make_pair(shot->pnt(), shot->spd()));
	}
}

#ifdef HAVE_LUA
void Game::addEnemy(const Point& pnt, const Point& spd,
					struct lua_State* state, const char* func,
					const Enemy* srcEne, int id)
{
	Enemy* enemy =
		new Enemy(pnt, spd, enemygZako_.get(), state, func,
				  !srcEne->blue(), id);
	enemy_.push_back(enemy);
	chars_.push_back(enemy);

	if (Conf::instance()->isCpu()) {
		cpuInput_->registEnemy(enemy);
	}
}
#endif

void Game::addEnemy(const Point& pnt, const Point& spd,	BulletMLState* state,
					const Enemy* srcEnemy)
{
	const std::vector<BulletMLNode*>& acts = state->getNode();
	std::vector<BulletMLNode*>::const_iterator ite;
	for (ite = acts.begin(); ite != acts.end(); ite++) {
		const BulletMLNode* act = *ite;
		if (act->findNode(BulletMLNode::fire)) break;
		if (act->findNode(BulletMLNode::fireRef)) break;
		if (act->findNode(BulletMLNode::actionRef)) break;
	}

	Enemy* enemy;
	if (ite != acts.end()) {
		enemy =
			new Enemy(pnt, spd, enemygZako_.get(), state, !srcEnemy->blue());
	}
	else {
		enemy =
			new Enemy(pnt, spd, enemygBullet_.get(), state, !srcEnemy->blue());
	}
	enemy_.push_back(enemy);
	chars_.push_back(enemy);


	if (Conf::instance()->isCpu()) {
		cpuInput_->registEnemy(enemy);
	}

}

#ifdef MOVIE
extern "C" {
#include "SDL_SaveMovie.h"
}
#endif

int Game::start() {
	timer_.start();
	start_ = true;
	end_ = false;

	PROFILE_INIT;
	{
		PROFILE("main loop");

#ifdef MOVIE
        SDL_SaveMovie_Init(screen_->getSurface(), "test.swf", "libSDL_SaveMovie_SWF.so", 100, 40);/* 0.1msec */
#endif

		while (!end_) {
			playerEv();
			enemyEv();
			charMoveValue();
			charWallEvent();
			charMoveGraphic();
			updateScreen();
			clearScreen();
			timerEv();
			hitEv();
			deadEv();
			endEv();
			levelEv();
			if (Conf::instance()->isReplay() && !replay_->hasNext()) {
				end_ = true;
			}
		} 
#ifdef MOVIE
		SDL_SaveMovie_Terminate();
#endif
    }
	PROFILE_OUTPUT;

	if (Conf::instance()->isCpu()) {
		cpuInput_->report();
	}

	processRanking();

	if (quit_ || Conf::instance()->isFastQuit()) return -1;

	start_ = false;

	waitInputRelease();
	if (waitAnyInput() != 0) {
		return -1;
	}

	mainFrame()->fill(Color::BLACK);
	if (Conf::instance()->mode() <= 1 ||
		(Conf::instance()->sp() &&
		 Conf::instance()->gameType() == Conf::SDMKUN))
	{
		select_->putRanking();
	}

	Game::instance()->waitInputRelease();
	if (select_->saveReplay() != 0) {
		return -1;
	}

	return 0;
}

void Game::processRanking() {
	if (Conf::instance()->sp()) {
		ellapsedSecOrRank_ = getLimit();
	}
	else {
		ellapsedSecOrRank_ = timer_.getEllapsedSecond();
	}

	if (nowBullet_ != 0) {
		// }WbNio[c
		if (Conf::instance()->mode() <= 1) {
			nowBullet_->doDamage();
		}
		if (Conf::instance()->sp()) {
			if (Conf::instance()->gameType() == Conf::SDMKUN) {
				nowBullet_ = bulletInfo_->descriptions()[0];
			}
			nowBullet_->updateLimit(ellapsedSecOrRank_);
		}
	}

	if (!Conf::instance()->isFastMode()) {
		bulletInfo_->saveUserInfo();
	}

	select_->insertRanking(ellapsedSecOrRank_, Conf::instance()->sp());
}

Game::Game()
{}

Game::~Game() {
	delete screen_;

	delete_clear(chars_);

	Kanji_CloseFont(font_);
}

void Game::timerEv() {
	PROFILE("timer");

	timer_.wait();

	if (bulletInfoTicks_ < timer_.getEllapsedTicks()) {
		bulletInfoTicks_ = 0;
	}

	if (bulletInfoTicks_ == 0 && screenIsLarge()) {
		if (Conf::instance()->sp()) {
			if (ellapsedSecOrRank_ != getLimit()) {
				ellapsedSecOrRank_ = getLimit();
				select_->putStatusLine(
					boost::lexical_cast<std::string>(ellapsedSecOrRank_) +
					"rank. ");
			}
		}
		else {
			if (ellapsedSecOrRank_ != timer_.getEllapsedSecond()) {
				ellapsedSecOrRank_ = timer_.getEllapsedSecond();
				select_->putStatusLine(
					boost::lexical_cast<std::string>(ellapsedSecOrRank_) +
					"sec. ");
			}
		}
	}

	int verbose = Conf::instance()->verbose();
	if (verbose) {
		if (timer_.getTurn()%verbose == 0) {
			std::cout << timer_.getTurn() << "turn. "
					  << shot_.size() << "shots. "
					  << enemy_.size() << "enemies. "
					  << std::endl;;
		}
	}
}

void Game::levelEv() {
	PROFILE("level");

	if (Conf::instance()->sp()) {
		if (getFrameTurn() % 30 == 29) {
			if (rank_ >= 1) {
				rank_ = 0;
				enemyNum_++;
				Enemy* enemy = new BossEnemy(Point(100+rnd(100), 50+rnd(100)),
											 enemygBoss_.get(),
											 enemyMgr_.getBehavior());

				enemy_.push_back(enemy);
				chars_.push_back(enemy);
			}
			rank_ += 0.01;
		}
	}
}

void Game::playerEv() {
	PROFILE("player");

	player_->keyEvent();
}

void Game::enemyEv() {
	PROFILE("enemy");

	// BulletML ǂނ̂͑Ƃ 60FPS ɂB
	int now = getFrameTurn();
	if (now != lastEnemyTurn_) {
		std::for_each(enemy_.begin(), enemy_.end(),
					  std::mem_fun(&Enemy::huntEvent));
		lastEnemyTurn_ = now;
	}
}

void Game::deadEv() {
	PROFILE("dead");

	std::list<Charactor*>::iterator ite;
	std::list<Shot*>::iterator ites;
	std::list<Enemy*>::iterator itee;

	for (ite = chars_.begin(); ite != chars_.end();) {
		Charactor* ch = *ite;

		if (!ch->alive()) {
			if (shot_.end() !=
				(ites = std::find(shot_.begin(), shot_.end(), ch)))
				shot_.erase(ites);
			else if (enemy_.end() !=
					 (itee = std::find(enemy_.begin(), enemy_.end(), ch))) {
				enemy_.erase(itee);
			}

			ite = chars_.erase(ite);
			delete ch;
		}
		else ite++;
	}
}

void Game::hitEv() {
	PROFILE("hit");

	std::list<Shot*>::iterator ites;
	for (ites = shot_.begin(); ites != shot_.end(); ites++) {
		Shot* sh = *ites;
		if (player_->isHit(sh)) {
			sh->hit(player_);
			player_->hit(sh);
		}
	}
	std::list<Enemy*>::iterator itee;
	for (itee = enemy_.begin(); itee != enemy_.end(); itee++) {
		Enemy* en = *itee;
		if (player_->isHit(en)) {
			en->hit(player_);
			player_->hit(en);
		}
	}
}

void Game::clearScreen() {
	PROFILE("clear");

	if (1) {
		std::list<Charactor*>::iterator ite;
		for (ite = chars_.begin(); ite != chars_.end(); ite++) {
			Charactor* ch = *ite;

			Rect rect(ch->pnt(), ch->size(), true);
			if (!timer_.shouldSkip()) {
				mainFrame()->fillRect(rect, Color::BLACK);
			}
		}
	}
	else {
		mainFrame()->fill(Color::BLACK);
	}
}

void Game::charMoveValue() {
	PROFILE("move value");

	std::for_each(chars_.begin(), chars_.end(),
				  std::mem_fun(&Charactor::moveValue));
}

void Game::charWallEvent() {
	PROFILE("wall event");

	std::for_each(chars_.begin(), chars_.end(),
				  std::mem_fun(&Charactor::wallEvent));
}

void Game::charMoveGraphic() {
	PROFILE("move graphic");

	if (!timer_.shouldSkip()) {
		std::for_each(chars_.begin(), chars_.end(),
					  std::mem_fun(&Charactor::moveGraphic));
	}
}

void Game::updateScreen() {
	PROFILE("update screen");

	/*
	  if (turn_%20 == 0) {
	  std::ostringstream oss;
	  oss << "dump/" << turn_ << ".bmp";
	  screen_->save(oss.str());
	  }
	*/

	if (!timer_.shouldSkip()) {
		screen_->flip();
	}
}

void Game::getEnemyPosAndSpd(std::vector<std::pair<Point, Point> >& out) const
{
	std::list<Enemy*>::const_iterator itee;
	for (itee = enemy_.begin(); itee != enemy_.end(); itee++) {
		const Enemy* ene = *itee;
		out.push_back(std::make_pair(ene->pnt(), ene->spd()));
	}
	std::list<Shot*>::const_iterator ites;
	for (ites = shot_.begin(); ites != shot_.end(); ites++) {
		const Shot* shot = *ites;
		out.push_back(std::make_pair(shot->pnt(), shot->spd()));
	}
}

void Game::switchScreenMode(bool large) {
	mainFrame_.reset(0);

	Uint32 flag = 0;
	if (Conf::instance()->fullScreen()) flag = SDL_FULLSCREEN;

	// ̂Ƃ傫ʂɂӖȂB
//	if (large /*&& !Conf::instance()->isFastMode()*/) {
	if (large && !Conf::instance()->isFastMode()) {
		screen_->resize(Conf::instance()->normalDispSize(), flag);
		mainFrame_.reset(new Surface(*screen_, Screen::MAIN_FRAME));
	}
	else {
		screen_->resize(Conf::instance()->miniDispSize(), flag);
	}
}

bool Game::screenIsLarge() const {
	return mainFrame_.get() != 0;
}

Surface* Game::mainFrame() {
	if (screenIsLarge()) {
		return mainFrame_.get();
	}
	else {
		return screen_;
	}
}

void Game::noticeChangeBullet(int index) {
	nowBullet_ = bulletInfo_->descriptions()[index];

	select_->putBulletInfo(nowBullet_);
	bulletInfoTicks_ = timer_.getEllapsedTicks() + 3000;

	// }WbNio[c
	if (Conf::instance()->mode() <= 1) {
		nowBullet_->doCome();
	}
}

void Game::saveReplay(const std::string& c) {
	std::string prefix(std::string("replay/")+c);
	replay_->save(prefix+"-input.dat", prefix+"-rnd.dat",
				  prefix+"-rndmax.dat", prefix+"-bullets.dat");

	std::ofstream os((prefix+"-name.dat").c_str());
	os << Conf::instance()->title() << std::endl;

	os << ellapsedSecOrRank_;
	if (Conf::instance()->sp()) os << "Rank";
	else os << "Sec";
}

void Game::loadReplay(const std::string& prefix) {
	replay_.reset(new Replay());
	replay_->load(prefix+"-input.dat", prefix+"-rnd.dat",
				  prefix+"-rndmax.dat", prefix+"-bullets.dat");

}
