#include "game.h"
#include "command.h"
#include "command_bulletml.h"
#include "player.h"
#include "bulletinfo.h"
#include "replay.h"
#include "conf.h"
#include "cpputil.h"
#include "usererror.h"
#include "strutil.h"

#ifdef HAVE_LUA
#include "command_lua.h"
#endif

#include <bulletml/bulletmlparser.h>
#include <bulletml/bulletmlparser-tinyxml.h>

#include <fstream>
#include <sstream>
#include <algorithm>
#include <functional>

#include <cstdio>
#include <cctype>

EnemyBehavior::EnemyBehavior(const std::string& filename)
    : commandType_(Random)
#ifdef HAVE_LUA
	, lua_(false)
#endif
{
    const Conf* conf = Conf::instance();
    if (conf->gameType() == Conf::XML) {
		std::string suf = filename.substr(filename.find_last_of("."));

#ifdef HAVE_LUA
		// it's not xml...
		if (suf == ".lua" || suf == ".LUA") {
			lua_ = true;
			filename_ = filename;
		}
		else {
#endif
			BulletMLParser* bs;

			if (suf == ".bml" || suf == ".BML") {
				FILE* fp = std::fopen(filename.c_str(), "r");
				check(fp != 0, filename + ": bml File not Found!\n");

				std::string first = my_getline(fp);
				if (first.find("<?xml") == 0) {
					TiXmlDocument doc(filename.c_str());
					doc.LoadFile();
					TiXmlNode *node, *node2;
					for (node = doc.FirstChild();
						 node; node = node->NextSibling())
					{
						if (node->ToElement() != 0) break;
					}
					check(node != 0, "<barrage> not found");
					for (node2 = node->FirstChild();
						 node2; node2 = node2->NextSibling())
					{
						TiXmlElement* elem = node2->ToElement();
						if (elem != 0) {
							if (elem->Value() == "bulletml") {
								bs = new BulletMLParserTinyXML(node2,filename);
								bs->build();
								break;
							}
							else if (elem->Value() == "lua") {
								for (TiXmlNode* n = elem->FirstChild();
									 n; n = n->NextSibling())
								{
									TiXmlText* text = n->ToText();
									if (text == 0) continue;
									luaScript_ = text->Value();
									lua_ = true;
									filename_ = filename;
								}
								break;
							}
						}
					}
					check(node2 != 0, "<bulletml> or <lua> not found");
				}
				else {
					for (int cnt = 0; cnt < 3; cnt++) {
						my_getline(fp);
						if (feof(fp)) {
							throw UserError("too short bml file");
						}
					}
					bs = new BulletMLParserTinyXML(fp, filename);
					bs->build();
				}
			}
			else {
				bs = new BulletMLParserTinyXML(filename);
				bs->build();
			}
			commands_.push_back(bs);
#ifdef HAVE_LUA
		}
#endif
    }
    else if (conf->gameType() == Conf::ACT) {
		std::ifstream is(filename.c_str());
		check(is.is_open(), filename + "action File not Found!\n");
		parseActionFile(is);
    }
	else {
		std::ifstream is("bosses.d/main.act");
		check(is.is_open(), "bosses.d/main.act: action File not Found!\n");
		parseActionFile(is);
	}
}

EnemyBehavior::~EnemyBehavior() {
	if (!lua_) delete_clear(commands_);
}

void EnemyBehavior::parseActionFile(std::istream& is) {
	const std::vector<BulletDescription*>& descs =
		Game::instance()->getBulletInfo()->descriptions();

    static const int MAXBUF = 256;
    char line[MAXBUF];

    while (!is.eof()) {
		is.getline(line, MAXBUF);
		std::istringstream iss(line);
		std::string command;
		iss >> command;

		ignoreBrankAndComment(command);
		if (command.empty()) continue;

		if (command == "Commands") {
			std::string args;
			iss >> args;
			if (args == "List") commandType_ = List;
			else if (args == "Random") commandType_ = Random;
			else check(0, "Unknown \"Commands\" argument" + args);
		}
		else {
			BulletMLParser* bs = new BulletMLParserTinyXML(command);
			bs->build();
			commands_.push_back(bs);

			for (size_t i = 1; i < descs.size(); i++) {
				if (command.find(descs[i]->file()) != std::string::npos) {
					numberMap_.push_back(i);
					break;
				}
			}
		}
    }

    init();
}

void EnemyBehavior::init() {
    commandIte_ = commands_.begin();
}

std::auto_ptr<EnemyCommand> EnemyBehavior::getRandomAction(Enemy* ene) {
#ifdef HAVE_LUA
	if (lua_) {
		if (luaScript_ != "") {
			return std::auto_ptr<EnemyCommand>(
				new EnemyCommandLua(filename_, luaScript_, ene));
		}
		else {
			return std::auto_ptr<EnemyCommand>(
				new EnemyCommandLua(filename_, ene));
		}
	}
#endif

	if (Conf::instance()->isReplay() && commands_.size() != 1) {
		num_ = rnd(Game::instance()->getReplay()->bullets());
	}
	else {
		num_ = rnd(static_cast<int>(commands_.size()));
	}
    return std::auto_ptr<EnemyCommand>(
		new EnemyCommandBulletML(commands_[num_], ene));
}

std::auto_ptr<EnemyCommand> EnemyBehavior::getNextAction(Enemy* ene) {
    BulletMLParser* ech = *commandIte_;
    commandIte_++;
    if (commandIte_ == commands_.end()) commandIte_ = commands_.begin();
    return std::auto_ptr<EnemyCommand>(
		new EnemyCommandBulletML(ech, ene));
}


int EnemyBehavior::getBulletNumber() {
	assert((size_t)num_ < numberMap_.size());
	return numberMap_[num_];
}

Enemy::Enemy (const Point& pnt, Surface* graph)
	: Charactor(pnt, graph), blue_(false)
{
}

BossEnemy::BossEnemy(const Point& pnt, Surface* graph, EnemyBehavior* behavior)
	: Enemy(pnt, graph), behavior_(behavior), firstPnt_(pnt_)
{
    behavior_->init();
	waitTime_ = Game::instance()->getFrameTurn() + 60;
	zako_ = false;
}

Enemy::Enemy (const Point& pnt, const Point& spd, Surface* graph,
			  BulletMLState* state, bool blue)
    : Charactor(pnt, graph), blue_(blue)
{
	com_.reset(new EnemyCommandBulletML(state, this));

    spd_ = spd;
    zako_ = true;
}

#ifdef HAVE_LUA
Enemy::Enemy (const Point& pnt, const Point& spd, Surface* graph,
	   struct lua_State* state, const char* func, bool blue, int id)
    : Charactor(pnt, graph), blue_(blue)
{
	com_.reset(new EnemyCommandLua(state, func, this, id));

    spd_ = spd;
    zako_ = true;
}
#endif

Enemy::~Enemy() {}

void Enemy::huntEvent() {
    if (com_.get() != 0) {
		com_->run();
		if (com_->isCommandEnd()) {
			com_.reset(0);
			endBullet();
		}
    }
	else {
		nextBullet();
	}
}

void BossEnemy::endBullet() {
	std::cout << Game::instance()->getFrameTurn() << " "<< SDL_GetTicks() << " end\n"; ///@@@@ debuging...
	waitTime_ = Game::instance()->getFrameTurn() + 60;
    spd_ = Point(0, 0);
}

void BossEnemy::nextBullet() {
	if (waitTime_ <= Game::instance()->getFrameTurn()) {
		if (behavior_->isRandomAction()) {
			com_.reset(behavior_->getRandomAction(this).release());
			if (Conf::instance()->isNormalMode()) {
				Game::instance()->noticeChangeBullet(
					behavior_->getBulletNumber());
			}
		}
		else {
			com_.reset(behavior_->getNextAction(this).release());
		}
		std::cout << Game::instance()->getFrameTurn() << " "<< SDL_GetTicks() << " start\n"; ///@@@@ debuging...
	}
	else {
		Point vec(firstPnt_ - pnt_);
        double len = vec.length();
		if (len > 3) {
			vec.unit();
            std::cout << pnt_.x << "," << pnt_.y << "=>";
			pnt_ += vec * std::max(7.0, len/20);
            std::cout << pnt_.x << "," << pnt_.y << std::endl;
		}
		else {
			pnt_ = firstPnt_;
		}
	}
}

void Enemy::hitByPlayer (Player*) {
}

void Enemy::hitByEnemy (Enemy*) {
}

void Enemy::hitByShot (Shot*) {
}

EnemyMgr::EnemyMgr() {}

void EnemyMgr::init() {
    behavior_.reset(new EnemyBehavior(Conf::instance()->actionFile()));
}

EnemyBehavior* EnemyMgr::getBehavior() {
	return behavior_.get();
}

