/**
 * $Id: command_lua.cc,v 1.4 2003/10/13 22:54:34 i Exp $
 *
 * Copyright (C) shinichiro.h <s31552@mail.ecc.u-tokyo.ac.jp>
 *  http://user.ecc.u-tokyo.ac.jp/~s31552/wp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#ifdef HAVE_LUA

#include "command_lua.h"
#include "usererror.h"
#include "enemy.h"
#include "game.h"
#include "player.h"

extern "C" {
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
}

#include <map>

void displayError(const std::string& msg, const std::string& kind);

namespace {
	Enemy* nowEnemy;
	EnemyCommandLua* nowCommand;
	std::map<lua_State*, int> refCount;

	int luaPanic(lua_State* lua) {
		fprintf(stderr, "errored %s\n", lua_tostring(lua, 0));
		exit(1);
	}

	void throwError(lua_State* , std::string msg) {
		std::cerr << msg << std::endl;
//		throw UserError(msg);
		displayError(msg, "User");
		exit(1);
	}
	void throwError(lua_State* lua, std::string msg, int index) {
		throwError(lua, msg + ": " + lua_tostring(lua, index));
	}

	double getDouble(lua_State* lua, int i) {
		if (!lua_isnumber(lua, i)) {
			throwError(lua, "incorrect argument");
		}
		return lua_tonumber(lua, i);
	}

	int getInt(lua_State* lua, int i) {
		return static_cast<int>(getDouble(lua, i));
	}

	const char* getString(lua_State* lua, int i) {
		if (!lua_isstring(lua, i)) {
			throwError(lua, "incorrect argument");
		}
		return lua_tostring(lua, i);
	}

	int fire(lua_State* lua) {
		int n = lua_gettop(lua);
		if (n == 2) {
			double sx = getDouble(lua, 1);
			double sy = getDouble(lua, 2);
			Game::instance()->addShot(nowEnemy->center(), Point(sx, sy) * 100,
									  nowEnemy);
		}
		else if (n == 3) {
			double sx = getDouble(lua, 1);
			double sy = getDouble(lua, 2);
			const char* func = getString(lua, 3);
			Game::instance()->addEnemy(nowEnemy->center(), Point(sx, sy) * 100,
									   lua, func, nowEnemy);
		}
		else if (n == 4) {
			double sx = getDouble(lua, 1);
			double sy = getDouble(lua, 2);
			const char* func = getString(lua, 3);
			int id = getInt(lua, 4);
			Game::instance()->addEnemy(nowEnemy->center(), Point(sx, sy) * 100,
									   lua, func, nowEnemy, id);
		}
		else {
			throwError(lua, "fire must have 2 or 3 or 4 arguments");
		}
		return 0;
	}
	int setPos(lua_State* lua) {
		int n = lua_gettop(lua);
		if (n != 2) {
			throwError(lua, "setPos must have 2 arguments");
		}
		nowCommand->setPos(getDouble(lua, 1), getDouble(lua, 2));
		return 0;
	}
	int setSpeed(lua_State* lua) {
		int n = lua_gettop(lua);
		if (n != 2) {
			throwError(lua, "setSpeed must have 2 arguments");
		}
		nowCommand->setSpeed(getDouble(lua, 1), getDouble(lua, 2));
		return 0;
	}
	int quit(lua_State*) {
		nowCommand->end();
		return 0;
	}
	int vanish(lua_State*) {
		nowCommand->vanish();
		return 0;
	}
	int getRank(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->getRank());
		return 1;
	}
	int getTurn(lua_State* lua) {
		lua_pushnumber(lua, nowCommand->getTurn());
		return 1;
	}
	int getX(lua_State* lua) {
		lua_pushnumber(lua, nowEnemy->center().x);
		return 1;
	}
	int getY(lua_State* lua) {
		lua_pushnumber(lua, nowEnemy->center().y);
		return 1;
	}
	int getSpeedX(lua_State* lua) {
		lua_pushnumber(lua, nowEnemy->spd().x * 0.01);
		return 1;
	}
	int getSpeedY(lua_State* lua) {
		lua_pushnumber(lua, nowEnemy->spd().y * 0.01);
		return 1;
	}
	int getPlayerAngle(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->player()->center().angle(nowEnemy->center()));
		return 1;
	}
	int getPlayerX(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->player()->center().x);
		return 1;
	}
	int getPlayerY(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->player()->center().y);
		return 1;
	}
	int getPlayerSpeedX(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->player()->spd().x * 0.01);
		return 1;
	}
	int getPlayerSpeedY(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->player()->spd().y * 0.01);
		return 1;
	}
	int getMaxX(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->mainFrame()->size().x);
		return 1;
	}
	int getMaxY(lua_State* lua) {
		lua_pushnumber(lua, Game::instance()->mainFrame()->size().y);
		return 1;
	}

	struct LuaRunner {
		LuaRunner(const std::string& l) : lua(l), end(false) {}
		const std::string& lua;
		bool end;
	};
	const char* chunkReader(lua_State*, void* data, size_t* size) {
		LuaRunner* lr = (LuaRunner*)data;
		if (lr->end) return 0;
		lr->end = true;
		*size = lr->lua.size();
		return lr->lua.c_str();
	}
}

EnemyCommandLua::EnemyCommandLua(std::string filename, Enemy* enemy)
	: enemy_(enemy), func_("top"), id_(-1), end_(false), turn_(0)
{
	std::cout << filename << " is next\n";

	lua_ = lua_open();

	refCount[lua_] = 1;

	lua_atpanic(lua_, luaPanic);

	lua_baselibopen(lua_);
	lua_mathlibopen(lua_);

	int ret = luaL_loadfile(lua_, filename.c_str());
	if (ret != 0) {
		if (ret == LUA_ERRSYNTAX)
			throwError(lua_, filename + ": syntax error", 0);
		else if (ret == LUA_ERRMEM)
			throwError(lua_, filename + ": memory error", 0);
		else
			throwError(lua_, filename + ": some error", 0);
	}

	init();
}

EnemyCommandLua::EnemyCommandLua(std::string filename,
								 const std::string& lua, Enemy* enemy)
	: enemy_(enemy), func_("top"), id_(-1), end_(false), turn_(0)
{
	lua_ = lua_open();

	refCount[lua_] = 1;

	lua_atpanic(lua_, luaPanic);

	lua_baselibopen(lua_);
	lua_mathlibopen(lua_);

	LuaRunner lr(lua);

	int ret = lua_load(lua_, chunkReader, &lr, filename.c_str());
	if (ret != 0) {
		if (ret == LUA_ERRSYNTAX)
			throwError(lua_, filename + ": syntax error", 0);
		else if (ret == LUA_ERRMEM)
			throwError(lua_, filename + ": memory error", 0);
		else
			throwError(lua_, filename + ": some error", 0);
	}

	init();
}

void EnemyCommandLua::init() {
	lua_pcall(lua_, 0, LUA_MULTRET, 0);

	lua_register(lua_, "fire", fire);
	lua_register(lua_, "quit", quit);
	lua_register(lua_, "vanish", ::vanish);
	lua_register(lua_, "setPos", ::setPos);
	lua_register(lua_, "setSpeed", ::setSpeed);
	lua_register(lua_, "getRank", getRank);
	lua_register(lua_, "getTurn", ::getTurn);
	lua_register(lua_, "getX", getX);
	lua_register(lua_, "getY", getY);
	lua_register(lua_, "getSpeedX", getSpeedX);
	lua_register(lua_, "getSpeedY", getSpeedY);
	lua_register(lua_, "getPlayerAngle", getPlayerAngle);
	lua_register(lua_, "getPlayerX", getPlayerX);
	lua_register(lua_, "getPlayerY", getPlayerY);
	lua_register(lua_, "getPlayerSpeedX", getPlayerSpeedX);
	lua_register(lua_, "getPlayerSpeedY", getPlayerSpeedY);
	lua_register(lua_, "getMaxX", getMaxX);
	lua_register(lua_, "getMaxY", getMaxY);
}

EnemyCommandLua::EnemyCommandLua(lua_State* state, const char* func,
								 Enemy* enemy, int id)
	: enemy_(enemy), func_(func), id_(id), end_(false), turn_(0)
{
	lua_ = state;
	++refCount[lua_];
}

EnemyCommandLua::~EnemyCommandLua() {
	if (--refCount[lua_] == 0) {
		lua_close(lua_);
	}
}

void EnemyCommandLua::run() {
	if (end_) return;

	nowCommand = this;
	nowEnemy = enemy_;
	lua_getglobal(lua_, func_.c_str());
	if (!lua_isfunction(lua_, -1)) {
		throwError(lua_, func_ + ": function not found");
	}

	int ret = 0;
	if (id_ == -1) {
		ret = lua_pcall(lua_, 0, 0, 0);
	}
	else {
		lua_pushnumber(lua_, id_);
		ret = lua_pcall(lua_, 1, 0, 0);
	}

	if (ret) {
		throwError(lua_, func_ + ": function errored", -1);
	}

	turn_++;
}

bool EnemyCommandLua::isCommandEnd() const {
    return end_;
}

void EnemyCommandLua::setPos(double x, double y) const {
	enemy_->center_ = Point(x, y);
	enemy_->pnt_ =
		enemy_->center_ - Point(enemy_->graph_->size().xi()>>1,
								enemy_->graph_->size().yi()>>1);
}

void EnemyCommandLua::setSpeed(double sx, double sy) const
{
	enemy_->spd_ = Point(sx * 100, sy * 100);
}

void EnemyCommandLua::vanish() const {
	enemy_->alive_ = false;
}

#endif // HAVE_LUA
