#ifndef BOHA_BULLET_INFO
#define BOHA_BULLET_INFO

#include "cpputil.h"
#include "config.h"

#include <string>
#include <vector>
#include <algorithm>
#include <memory>
#include <map>
#include <iosfwd>

class Surface;

class BulletDescription {
public:
	explicit BulletDescription(std::istream& is);
	BulletDescription(std::ifstream& is, const std::string& file,
					  int id, class BulletInfo* info);
	BulletDescription(const std::string& file, const std::string& title,
					  int id);

	int id() const { return id_; }
	const std::string& file() const { return file_; }
	const std::string& title() const { return title_; }
	const std::string& description() const { return description_; }
	const std::string& capture() const { return capture_; }
	const std::string& group() const { return group_; }
	std::string xmlFile() const;
	bool isUserDefined() const { return userDefined_; }

	void loadUserInfo(const std::string& userName);
	void saveUserInfo(const std::string& userName) const;
	void doCome() { come_++; }
	void doDamage() { damage_++; }
	void updateLimit(int limit) { limit_ = std::max(limit, limit_); }
	int getCome() const { return come_; }
	int getDamage() const { return damage_; }
	int getSuccessPer() const {
		if (come_ == 0) return 0;
		return (come_-damage_)*100/come_;
	}
	int getLimit() const { return limit_; }
	Surface* shot() { return shot_.get(); }
	bool hasShot() const { return hasShot_; }

	bool isIdLess(const BulletDescription* rhs) const {
		return id_ < rhs->id_;
	}

	bool eval() const { return eval_; }
	int difficulty() const { return difficulty_; }
	int speed() const { return speed_; }
	int density() const { return density_; }
	int aim() const { return aim_; }
	int tricky() const { return tricky_; }

#ifdef HAVE_LUA
	bool isLua() const { return lua_; }
#endif

private:
	int id_;
	std::string file_, title_, description_, capture_, group_;
	std::auto_ptr<Surface> shot_;
	bool hasShot_;

	int come_, damage_, limit_;

	bool eval_;
	int difficulty_, speed_, density_, aim_, tricky_;

	bool userDefined_;

#ifdef HAVE_LUA
	bool lua_;
#endif

};

class GroupInfo {
public:
	GroupInfo();
	~GroupInfo();

	const std::vector<std::string>& getGroups() const {
		return groups_;
	}
	const std::string& getName(const std::string& sysname) const {
		GroupMap_::const_iterator ite = groupMap_.find(sysname);
		check(ite != groupMap_.end(), "group key " + sysname + " not found");
		return ite->second->name;
	}
	bool hasName(const std::string& sysname) const {
		GroupMap_::const_iterator ite = groupMap_.find(sysname);
		return ite != groupMap_.end();
	}

private:
	struct GroupDescription {
		GroupDescription() {}
 		std::string name;
	};

	typedef std::map<std::string, GroupDescription*> GroupMap_;
	GroupMap_ groupMap_;
	std::vector<std::string> groups_;

};

class BulletInfo {
public:
	typedef std::vector<BulletDescription*> Descriptions;

	BulletInfo();
	~BulletInfo();

	const Descriptions& descriptionsDisplayOrder() const {
		return descriptionsDisplayOrder_;
	}
	const Descriptions& descriptions() const {
		return descriptionsIdOrder_;
	}

	const std::vector<std::string>& groups() const {
		return groupInfo_.getGroups();
	}

	Descriptions getGroupBullets(const std::string& groupSysname) const;

	const std::string& getGroupName(const std::string& groupSysname) const {
		return groupInfo_.getName(groupSysname);
	}
	bool hasGroupName(const std::string& groupSysname) const {
		return groupInfo_.hasName(groupSysname);
	}

	void loadUserInfo(const std::string& userName);
	void saveUserInfo() const;

	int getBulletNum() const { return officialBulletNum_; }

private:
	Descriptions descriptionsDisplayOrder_;
	Descriptions descriptionsIdOrder_;

	std::string userName_;

	int officialBulletNum_;

	GroupInfo groupInfo_;

};

#endif // ! BOHA_BULLET_INFO
