#include "bulletinfo.h"
#include "cpputil.h"
#include "surface.h"
#include "conf.h"
#include "usererror.h"
#include "strutil.h"

#include <stdio.h>
#include <fstream>
#include <cassert>
#include <boost/functional.hpp>
#include <boost/compose.hpp>
#include <boost/lexical_cast.hpp>

#include "bulletml/tinyxml/tinyxml.h"

#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif

namespace {
	std::string bml_getline(std::istream& is, std::string errorMsg) {
		std::string ret = my_getline(is);
		if (is.eof()) throw UserError(errorMsg);
		return ret;
	}

	std::string get_text(TiXmlElement* node) {
		TiXmlNode* child;
		for (child = node->FirstChild(); child; child = child->NextSibling()) {
			TiXmlText* text;
			if ((text = child->ToText()) != 0) {
				return text->Value();
			}
		}
		return "";
	}
	std::string get_attr_text(TiXmlElement* node, std::string name) {
		TiXmlAttribute* attr;
		for (attr = node->FirstAttribute(); attr; attr = attr->Next()) {
			if (attr->Name() == name) return attr->Value();
		}
		return "";
	}
}

void BulletDescription::loadUserInfo(const std::string& userName) {
	std::ifstream is(("save/"+userName+file_+".dat").c_str());
	if (is.is_open()) {
		is >> come_ >> damage_ >> limit_;
	}
	else {
		come_ = damage_ = limit_ = 0;
	}
}

void BulletDescription::saveUserInfo(const std::string& userName) const {
	std::ofstream os(("save/"+userName+file_+".dat").c_str());
	os << come_ << std::endl
	   << damage_ << std::endl
	   << limit_ << std::endl;
}

void BulletInfo::loadUserInfo(const std::string& userName) {
	userName_ = userName;
	std::for_each(descriptionsIdOrder_.begin(), descriptionsIdOrder_.end(),
				  boost::bind2nd(
					  boost::mem_fun(&BulletDescription::loadUserInfo),
					  userName_));
}

void BulletInfo::saveUserInfo() const {
	std::for_each(descriptionsIdOrder_.begin(), descriptionsIdOrder_.end(),
				  boost::bind2nd(
					  boost::mem_fun(&BulletDescription::saveUserInfo),
					  userName_));
}

std::string BulletDescription::xmlFile() const {
	if (userDefined_) {
		return file();
	}
	else {
#ifdef HAVE_LUA
		if (lua_) return "bosses.d/" + file() + ".lua";
		else
#endif
			return "bosses.d/" + file() + ".xml";
	}
}

BulletDescription::BulletDescription(const std::string& file,
									 const std::string& title, int id)
	:  id_(id), file_(file), title_(title), group_("user"),
	   hasShot_(false), come_(0), damage_(0), limit_(0), eval_(false),
	   userDefined_(true)
#ifdef HAVE_LUA
     , lua_(false)
#endif
{}

BulletDescription::BulletDescription(std::ifstream& is,
									 const std::string& file, int id,
									 BulletInfo* info)
	: id_(id), file_(file), hasShot_(false),
	  come_(0), damage_(0), limit_(0), eval_(false), userDefined_(true)
#ifdef HAVE_LUA
	, lua_(false)
#endif
{
	std::string first = bml_getline(is, "bml file does'nt contain group");
	if (first.find("<?xml") == 0) {
		is.close();
		TiXmlDocument doc(file_.c_str());
		doc.LoadFile();
		TiXmlNode *node, *node2;
		for (node = doc.FirstChild();
			 node; node = node->NextSibling())
		{
			if (node->ToElement() != 0) break;
		}
		check(node != 0, "<barrage> not found");
		for (node2 = node->FirstChild();
			 node2; node2 = node2->NextSibling())
		{
			TiXmlElement* elem = node2->ToElement();
			if (elem != 0) {
				if (elem->Value() == "barrageInfo") break;
			}
		}
		if (node2 == 0) {
			std::cout << file_ << ": no barrage info" << std::endl;
		}
		else {
			for (TiXmlNode* node3 = node2->FirstChild(); node3;
				 node3 = node3->NextSibling())
			{
				TiXmlElement* elem = node3->ToElement();
				if (elem != 0) {
					if (elem->Value() == "filename") {
						std::string fn = "user/" + get_text(elem) + ".bml";
						if (fn != file_) {
							std::cout << "rename " + file_ + " to " + fn
									  << std::endl;
							std::ifstream chk(fn.c_str());
							if (chk.is_open()) {
								chk.close();
								if (remove(fn.c_str()) != 0) {
									perror("remove failed in BulletDescription");
									error("remove " + fn + " failed");
								}
							}
							if (rename(file_.c_str(), fn.c_str()) != 0) {
								perror("rename failed in BulletDescription");
								error("rename " + file_ +
									  " to " + fn + " failed");
							}
							if (Conf::instance()->actionFile() == file_) {
								Conf::instance()->setActionFile(fn);
							}
							file_ = fn;
						}
					}
					else if (elem->Value() == "group") {
						group_ = get_text(elem);
						if (!info->hasGroupName(group_))
							group_ = "user";
					}
					else if (elem->Value() == "title")
						title_ = get_text(elem);
					else if (elem->Value() == "description")
						description_ = get_text(elem);
					else if (elem->Value() == "capture")
						capture_ = get_text(elem);
				}
			}
		}
		for (node2 = node->FirstChild();
			 node2; node2 = node2->NextSibling())
		{
			TiXmlElement* elem = node2->ToElement();
			if (elem != 0) {
				if (elem->Value() == "barrageStyle") break;
			}
		}
		if (node2 == 0) {
			std::cout << file_ << ": no barrage style" << std::endl;
		}
		else {
			for (TiXmlNode* node3 = node2->FirstChild(); node3;
				 node3 = node3->NextSibling())
			{
				TiXmlElement* elem = node3->ToElement();
				if (elem != 0) {
					if (elem->Value() == "bulletLanguage") {
						std::string lang = get_attr_text(elem, "name");
						if (lang == "lua") {
							lua_ = true;
						}
						else if (lang != "bulletml") {
							throw UserError(lang + ": unknown language");
						}
					}
				}
			}
		}
	}
	else {
		group_ = first;
		title_ = bml_getline(is, "bml file does'nt contain title");
		description_ = bml_getline(is, "bml file does'nt contain description");
		capture_ = bml_getline(is, "bml file does'nt contain capture");
	}

#ifdef HAVE_LUA
	size_t i = group_.find(":lua");
	if (i != std::string::npos) {
		group_ = group_.substr(0, i);
		lua_ = true;
	}
#endif
}

BulletDescription::BulletDescription(std::istream& is)
	: come_(0), damage_(0), limit_(0), eval_(false), userDefined_(false)
#ifdef HAVE_LUA
	, lua_(false)
#endif
{
	static const int MAXBUF = 1000;

	char ln[MAXBUF];
	is.getline(ln, MAXBUF);
	if (ln[0] == '\0') return;
	id_ = boost::lexical_cast<int>(ln);
	is.getline(ln, MAXBUF);
	group_ = ln;
	is.getline(ln, MAXBUF);
	file_ = ln;
	is.getline(ln, MAXBUF);
	title_ = ln;
	is.getline(ln, MAXBUF);
	description_ = ln;
	is.getline(ln, MAXBUF);
	capture_ = ln;

#ifdef HAVE_LUA
	size_t i = group_.find(":lua");
	if (i != std::string::npos) {
		group_ = group_.substr(0, i);
		lua_ = true;
	}
#endif

	if (file_ != "main" && !file_.empty()) {
		std::string bmp("shots/" + file_ + ".bmp");
		if (std::ifstream(bmp.c_str()).is_open()) {
			hasShot_ = true;
			if (Conf::instance()->preloadShots()) {
				shot_.reset(new Surface(bmp));
			}
		}
		else {
			hasShot_ = false;
			std::cerr << "warn: " + bmp + " not found." << std::endl;
		}
	}

	std::ifstream ifs(("inspection/" + file() + ".dat").c_str());
	if (ifs.is_open()) {
		std::string dummy;
		eval_ = true;
		ifs >> dummy >> difficulty_;
		ifs >> dummy >> density_;
		ifs >> dummy >> speed_;
		ifs >> dummy >> aim_;
		ifs >> dummy >> tricky_;
	}
}

GroupInfo::GroupInfo() {
	static const int MAXBUF = 1000;

	std::ifstream is("data/groups.dat");
	check(is.is_open(), "data/groups.dat is not found.");

	while (!is.eof()) {
		std::string sysname, name;
		GroupDescription* gdesc = new GroupDescription();
		char sc[MAXBUF], nc[MAXBUF];
		is >> sc >> nc;
		sysname = sc;
		name = nc;
		gdesc->name = name;
		groupMap_.insert(std::make_pair(sysname, gdesc));

		groups_.push_back(sysname);
	}
}

GroupInfo::~GroupInfo() {
	delete_clear_2nd(groupMap_);
}

BulletInfo::BulletInfo() {
	std::ifstream is("data/bosses.dat");
	check(is.is_open(), "data/bosses.dat is not found.");

	int i = 0;
	while (!is.eof()) {
		try {
			BulletDescription* desc = new BulletDescription(is);
			if (desc->capture() == "") {
				delete desc;
				break;
			}
			descriptionsDisplayOrder_.push_back(desc);
		}
		catch (...) {
			throw std::runtime_error("boss data file is broken in line "
				+ boost::lexical_cast<std::string>(i));
		}

		++i;
	}

	officialBulletNum_ = i - 1;

	descriptionsIdOrder_.assign(descriptionsDisplayOrder_.begin(),
							   descriptionsDisplayOrder_.end());
	std::sort(descriptionsIdOrder_.begin(),
			  descriptionsIdOrder_.end(),
			  std::mem_fun(&BulletDescription::isIdLess));

#ifdef HAVE_DIRENT_H

	DIR* dir = opendir("user");
	int j = 0;
	while (dirent* d = readdir(dir)) {
		std::string name(std::string("user/") + d->d_name);

		bool isXml = (name.find(".xml") == name.size() - 4 ||
					  name.find(".lua") == name.size() - 4);
		bool isBml = (name.find(".bml") == name.size() - 4);
		if (isXml || isBml) {
			BulletDescription* desc;
			if (isBml) {
				std::ifstream bmlis(name.c_str());
				desc = new BulletDescription(bmlis, name, i, this);
			}
			else {
				std::string title(std::string("[U`")
								  + boost::lexical_cast<std::string>(++j));
				desc = new BulletDescription(name, title, i);
			}
			descriptionsDisplayOrder_.push_back(desc);
			descriptionsIdOrder_.push_back(desc);

			std::cout << name << ": user defined bullet" << std::endl;
			++i;
		}
	}

#endif // HAVE_DIRENT_H

	std::cout << "registered " << i-1 << " bullets." << std::endl;
}

BulletInfo::~BulletInfo() {
	delete_clear(descriptionsIdOrder_);
}

BulletInfo::Descriptions
BulletInfo::getGroupBullets(const std::string& groupSysname) const {
	Descriptions ret;
	std::remove_copy_if(
		descriptionsDisplayOrder_.begin(), descriptionsDisplayOrder_.end(),
		std::back_inserter(ret),
		boost::compose_f_gx(
			std::bind2nd(std::not_equal_to<std::string>(), groupSysname),
			std::mem_fun(&BulletDescription::group)));
	return ret;
}
