#include "SDL_kanji.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <SDL_endian.h>

#define BUF 256

Kanji_Font* Kanji_OpenFont(const char* file, int size) {
	Kanji_Font* font;
	font = (Kanji_Font*)malloc(sizeof(Kanji_Font));

	font->k_size = size;
	font->a_size = size / 2;
	font->sys = KANJI_JIS;

	if (0 == Kanji_AddFont(font, file)) {
		return font;
	}
	else {
		free(font);
		return 0;
	}
}

static int ParseFont(Kanji_Font* font, FILE* fp) {
	int i, n;
	unsigned char length, temp;
	unsigned char* p;

	fread(&font->k_size, sizeof(int), 1, fp);
	font->k_size=SDL_SwapLE32(font->k_size);
	font->a_size = font->k_size >> 1;

	n = 0;
	while (n < font->k_size) { n+=8; }
	font->char_byte = n * font->k_size >> 3;
	font->one_line_byte = n >> 3;

	p = font->moji = (unsigned char*)malloc(
		sizeof(unsigned char)*(96*96+256)*font->char_byte);

	n = 0;
	while(fread(&length, sizeof(unsigned char), 1, fp)!=0) {n++;
		if(length<0x80) {
			for(i=0; i<length+1; ++i) {
				fread(&temp, sizeof(unsigned char), 1, fp);n++;
				*p = temp;
				p++;
			}
		}
		else {
			fread(&temp, sizeof(unsigned char), 1, fp);n++;

			for(i=0x80; i<length+1; ++i) {
				*p = temp;
				p++;
			}
		}
	}

	return 0;
}

int Kanji_AddFont(Kanji_Font* font, const char* file) {
	FILE* fp;

	fp = fopen(file, "rb");
	if(fp==NULL){
		fprintf(stderr, "cant open [%s]\n", file);
		return -1;
	}

	if (0 != ParseFont(font, fp)) return -1;

	fclose(fp);
	return 0;
}

int Kanji_FontHeight(Kanji_Font* font) {
	return font->k_size;
}

int Kanji_FontWidth(Kanji_Font* font, const char* text) {
	if (text == 0) return font->a_size;
	else return strlen(text) * font->a_size;
}

static void euc2jis(unsigned char *c1, unsigned char *c2)
{
	*c1 &= 0x7f;
	*c2 &= 0x7f;
}

static void sjis2jis(unsigned char *c1, unsigned char *c2)
{
	if( *c2 < 0x9f )
	{
		if( *c1 < 0xa0 )
		{
			*c1 -= 0x81;
			*c1 *= 2;
			*c1 += 0x21;
		}
		else
		{
			*c1 -= 0xe0;
			*c1 *= 2;
			*c1 += 0x5f;
		}
		if( *c2 > 0x7f )
			-- *c2;
		*c2 -= 0x1f;
	}
	else
	{
		if( *c1 < 0xa0 )
		{
			*c1 -= 0x81;
			*c1 *= 2;
			*c1 += 0x22;
		}
		else
		{
			*c1 -= 0xe0;
			*c1 *= 2;
			*c1 += 0x60;
		}
		*c2 -= 0x7e;
	}
}

static void ConvertCodingSystem(Kanji_Font* font,
								unsigned char *c1, unsigned char *c2) {
	if (font->sys == KANJI_SJIS) {
		sjis2jis(c1, c2);
	}
	else if (font->sys == KANJI_EUC) {
		euc2jis(c1, c2);
	}
}

int Kanji_PutText(Kanji_Font* font, int dx, int dy,
				  SDL_Surface* dst, const char* txt, SDL_Color fg)
{
	Uint32 fgcol;
	int index;
	int x, y, cx = dx, cy = dy;
	unsigned char high, low;
	int minx, miny, maxx, maxy;
	int nowKanji = 0;
	const unsigned char* text = (const unsigned char*)txt;
	unsigned char* px;
	unsigned char* py;
	unsigned char now;
	unsigned char bitCnt = 0;
	Uint8* p;
	Uint8* pf;
	int bpp = dst->format->BytesPerPixel;

	if(SDL_MUSTLOCK(dst)){
		if (SDL_LockSurface(dst) < 0) return -1;
	}

	fgcol = SDL_MapRGB(dst->format, fg.r, fg.g, fg.b);
	while (*text != 0) {
		if (font->sys == KANJI_JIS && *text == 0x1b) {
			if (*(text+1) == 0x24 && *(text+2) == 0x42) {
				nowKanji = 1;
			}
			else if (*(text+1) == 0x28 && *(text+2) == 0x42) {
				nowKanji = 0;
			}
			text += 3;
			continue;
		}
		if (font->sys != KANJI_JIS) nowKanji = !isprint(*text);

		if (!nowKanji) {
			index = *text;
			text++;

			pf=(Uint8*)(dst->pixels)+cy*dst->pitch+cx*bpp;

			minx = miny = 0;
			maxx = font->a_size;
			maxy = font->k_size;

			py = &font->moji[index*font->char_byte];
			py += font->one_line_byte * (font->k_size - 1);

#define RENDER_LOOP_ASCII(pixel_type) \
			for (y = miny; y <= maxy; y++) { \
                p = pf; \
				px = py; \
				now = *px; \
				bitCnt = (1 << 7); \
				/* G */ \
				for (x = 0; x < font->a_size; x++) { \
					bitCnt >>= 1; \
					if (bitCnt == 0) { \
						px++; \
						now = *px; \
						bitCnt = (1 << 7); \
					} \
				} \
				for (x = minx; x < maxx; x++) { \
					if (now & bitCnt) { \
						*((pixel_type *)p)=(pixel_type)fgcol; \
					} \
                    p = (Uint8*)(((pixel_type*)p)+1); \
					bitCnt >>= 1; \
					if (bitCnt == 0) { \
						px++; \
						now = *px; \
						bitCnt = (1 << 7); \
					} \
				} \
				py -= font->one_line_byte; \
                pf += dst->pitch; \
			}

			switch (bpp) {
			case 1:
				RENDER_LOOP_ASCII(Uint8);
				break;
			case 2:
				RENDER_LOOP_ASCII(Uint16);
				break;
			case 4:
				RENDER_LOOP_ASCII(Uint32);
				break;
			}

			cx += font->a_size;
		}
		else {
			high = *text;
			low = *(text+1);
			ConvertCodingSystem(font, &high, &low);
			index = (high - 0x20) * 96 + low - 0x20 + 0xff;
			text += 2;

			pf=(Uint8*)(dst->pixels)+cy*dst->pitch+cx*bpp;

			minx = miny = 0;
			maxx = font->k_size;
			maxy = font->k_size;

			py = &font->moji[index*font->char_byte];
			py += font->one_line_byte * (font->k_size - 1);

#define RENDER_LOOP_KANJI(pixel_type) \
			for (y = miny; y <= maxy; y++) { \
                p = pf; \
				px = py; \
				now = *px; \
				bitCnt = (1 << 7); \
				for (x = minx; x < maxx; x++) { \
					if (now & bitCnt) { \
						*((pixel_type *)p)=(pixel_type)fgcol; \
					} \
                    p = (Uint8*)(((pixel_type*)p)+1); \
					bitCnt >>= 1; \
					if (bitCnt == 0) { \
						px++; \
						now = *px; \
						bitCnt = (1 << 7); \
					} \
				} \
				py -= font->one_line_byte; \
                pf += dst->pitch; \
			}

			switch (bpp) {
			case 1:
				RENDER_LOOP_KANJI(Uint8);
				break;
			case 2:
				RENDER_LOOP_KANJI(Uint16);
				break;
			case 4:
				RENDER_LOOP_KANJI(Uint32);
				break;
			}

			cx += font->k_size;
		}
	}

	if(SDL_MUSTLOCK(dst)){
		SDL_UnlockSurface(dst);
	}

	return 0;
}

SDL_Surface* Kanji_CreateSurface(Kanji_Font* font, const char* text,
								 SDL_Color fg, int bpp)
{
	SDL_Surface* textbuf;
	int len;
	Uint32 bgcol;

	if (text == NULL) return NULL;
	if (*text == 0) return NULL;
	len = strlen(text);

	textbuf = SDL_CreateRGBSurface(SDL_SWSURFACE,
								   font->a_size*len, font->k_size,
								   bpp, 0, 0, 0, 0);
	if (textbuf == NULL) {
		fprintf(stderr,"ERROR: at Kanji_RenderText\n");
		return NULL;
	}
	bgcol = SDL_MapRGB(textbuf->format, 255-fg.r, 255-fg.g, 255-fg.b);
	SDL_FillRect(textbuf, NULL, bgcol);
	SDL_SetColorKey(textbuf, SDL_SRCCOLORKEY, bgcol);

	Kanji_PutText(font, 0, 0, textbuf, text, fg);

	return textbuf;
}

void Kanji_CloseFont(Kanji_Font* font) {
	free(font->moji);
	free(font);
}

void Kanji_SetCodingSystem(Kanji_Font* font, Kanji_CodingSystem sys) {
	font->sys = sys;
}

